unit ExceptDlg;

// Usage: add the unit to VCL.NET Application 

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, System.ComponentModel, StdCtrls, ExtCtrls, Clipbrd,
  Borland.Vcl.Buttons, Borland.Vcl.ImgList;

type
  TExceptionDialog = class(TForm)
    ImageIcon: TImage;
    ButtonContinue: TButton;
    ButtonQuit: TButton;
    MemoExceptionMessage: TMemo;
    MemoDetail: TMemo;
    ImageList: TImageList;
    ButtonDetail: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure ButtonDetailsClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: UInt16;
      Shift: TShiftState);
  private
    FDetailsVisible: Boolean;
    FFullHeight: Integer;
    class var ExceptionShowing: Boolean;
    class procedure ExceptionHandler(Sender: TObject; E: Exception); static;
    class procedure ShowException(E: Exception); static;
    procedure SetDetailsVisible(const Value: Boolean);
  protected
    procedure CopyReportToClipboard;
    procedure DetailSeparator(const SectionName: string);
    procedure PopulateControls(E: Exception);
  public
    class function ExtractException(E: Exception): Exception;
    property DetailsVisible: Boolean read FDetailsVisible write SetDetailsVisible;
  end;

var
  ExceptionDialog: TExceptionDialog;

implementation

uses
  System.Reflection;

{$R *.nfm}

procedure TExceptionDialog.ButtonDetailsClick(Sender: TObject);
begin
  DetailsVisible := not DetailsVisible;
end;

procedure TExceptionDialog.CopyReportToClipboard;
var
  SL: TStringList;
begin
  SL := TStringList.Create;
  try
    SL.AddStrings(MemoExceptionMessage.Lines);
    SL.Add('');
    SL.AddStrings(MemoDetail.Lines);
    Clipboard.AsText := SL.Text;
  finally
    SL.Free;
  end;
end;

procedure TExceptionDialog.DetailSeparator(const SectionName: string);
begin
  MemoDetail.Lines.Add(SectionName + ' ' + StringOfChar('=', 80 - Length(SectionName)));
end;

class procedure TExceptionDialog.ExceptionHandler(Sender: TObject; E: Exception);
begin
  if ExceptionShowing then
    Application.ShowException(E)
  else
  begin
    ExceptionShowing := True;
    try
      ShowException(E);
    finally
      ExceptionShowing := False;
    end;
  end;
end;

class function TExceptionDialog.ExtractException(E: Exception): Exception;
begin
  Result := E;
  while Result.GetBaseException <> Result do
    Result := Result.GetBaseException;
end;

procedure TExceptionDialog.FormCreate(Sender: TObject);
begin
  ImageIcon.Picture.Icon.Handle := LoadIcon(0, IDI_ERROR);
end;

procedure TExceptionDialog.FormKeyDown(Sender: TObject; var Key: UInt16; Shift: TShiftState);
begin
  if (Key = Ord('C')) and (ssCtrl in Shift) then
  begin
    CopyReportToClipboard;
    MessageBeep(MB_OK);
  end;
end;

procedure TExceptionDialog.FormShow(Sender: TObject);
begin
  FFullHeight := ClientHeight;
  MessageBeep(MB_ICONERROR);
  DetailsVisible := False;
end;

procedure TExceptionDialog.PopulateControls(E: Exception);
var
  CurrentDomain: AppDomain;
  Assemblies: array of Assembly;
  A: Assembly;
  I: Integer;
begin
  MemoExceptionMessage.Text := AdjustLineBreaks(E.Message);
  MemoDetail.Lines.Clear;
  DetailSeparator('Stack trace');
  MemoDetail.Lines.Add(E.ClassInfo.ToString);
  MemoDetail.Lines.Add(E.StackTrace);
  CurrentDomain := AppDomain.CurrentDomain;
  Assemblies := CurrentDomain.GetAssemblies;
  MemoDetail.Lines.Add('');
  DetailSeparator('Loaded assemblies');
  for I := Low(Assemblies) to High(Assemblies) do
  begin
    A := Assemblies[I];
    MemoDetail.Lines.Add(A.GetName.Name);
    MemoDetail.Lines.Add(Format('  Assembly version: %s', [A.GetName.Version.ToString]));
    MemoDetail.Lines.Add(Format('  Win32 version: %s', [A.ImageRuntimeVersion]));
    MemoDetail.Lines.Add(Format('  Codebase: %s', [A.GetName.CodeBase]));
    MemoDetail.Lines.Add('----------');
  end;
end;

procedure TExceptionDialog.SetDetailsVisible(const Value: Boolean);

  procedure ButtonImageIndex(I: Integer);
  var
    B: TBitmap;
  begin
    B := TBitmap.Create;
    try
      ImageList.GetBitmap(I, B);
      ButtonDetail.Glyph.Assign(B);
    finally
      B.Free;
    end;
  end;

begin
  FDetailsVisible := Value;
  if Value then
  begin
    ClientHeight := FFullHeight;
    MemoDetail.Height := FFullHeight - MemoDetail.Top - 4;
    MemoDetail.Visible := True;
    ButtonImageIndex(0);
  end
  else
  begin
    MemoDetail.Visible := False;
    ClientHeight := MemoDetail.Top - 4;
    ButtonImageIndex(1);
  end;
end;

class procedure TExceptionDialog.ShowException(E: Exception);
begin
  if GetCapture <> 0 then SendMessage(GetCapture, WM_CANCELMODE, 0, 0);
  if ExceptionDialog = nil then
    ExceptionDialog := TExceptionDialog.Create(Application);
  try
    ExceptionDialog.Caption := Application.Title;
    ExceptionDialog.PopulateControls(ExtractException(E));
    if ExceptionDialog.ShowModal = mrAbort then
      Halt(0);
  finally
    FreeAndNil(ExceptionDialog);
  end;
end;

procedure InitializeHandler;
begin
  Application.OnException := TExceptionDialog.ExceptionHandler;
end;

initialization
  InitializeHandler;

end.
